#include <Adafruit_GFX.h>
#include <Adafruit_ILI9341.h>
#include <SPI.h>
#include "spas.h"

#define BTN_PIN       12
#define POT_PIN       34
#define TFT_DC        2
#define TFT_CS        15
#define TFT_RST       4

Adafruit_ILI9341 tft(TFT_CS, TFT_DC, TFT_RST);

// === УПРАВЛЕНИЕ ИЗОБРАЖЕНИЯМИ ===
bool showClock = false;          // false = башня, true = часы
bool lastBtnState = HIGH;
unsigned long lastDebounce = 0;
const int DEBOUNCE_DELAY = 200;  // Увеличил для надёжности

// === СНЕЖИНКИ ===
struct Snowflake {
  float x, y;
  float speed;
  bool active;
};

#define MAX_SNOW 60
#define SNOW_COLOR 0xFFFF
Snowflake snow[MAX_SNOW];
int activeSnowCount = 0;

void setup() {
  pinMode(BTN_PIN, INPUT_PULLUP);
  pinMode(POT_PIN, INPUT);
  Serial.begin(115200);
  
  tft.begin();
  tft.setRotation(0);
  
  // Инициализация снежинок
  for (int i = 0; i < MAX_SNOW; i++) {
    snow[i].active = false;
  }
  
  drawCurrentImage();
  Serial.println("Готово! Кнопка меняет фон, потенциометр - снежинки");
}

void loop() {
  handleButton();
  updateSnow();
  delay(20);
}

// === 1. КНОПКА - КОРОТКОЕ НАЖАТИЕ МЕНЯЕТ ФОН ===
void handleButton() {
  int btnState = digitalRead(BTN_PIN);
  
  // Если кнопка нажата (LOW) и до этого была отпущена
  if (btnState == LOW && lastBtnState == HIGH) {
    lastDebounce = millis();
  }
  
  // Если кнопка отпущена (HIGH) и до этого была нажата
  if (btnState == HIGH && lastBtnState == LOW) {
    // Проверяем, что это было короткое нажатие (50-500 мс)
    if (millis() - lastDebounce > 50 && millis() - lastDebounce < 500) {
      showClock = !showClock; // Меняем фон
      
      Serial.print("Короткое нажатие! Фон: ");
      Serial.println(showClock ? "ЧАСЫ" : "БАШНЯ");
      
      // 1. Стираем ВСЕ снежинки
      clearAllSnowflakes();
      
      // 2. Рисуем новую картинку
      drawCurrentImage();
      
      // 3. Пересоздаём снежинки (они начнут падать сверху)
      resetSnowflakes();
    }
  }
  
  lastBtnState = btnState;
}

// Стираем все снежинки (восстанавливаем фон)
void clearAllSnowflakes() {
  for (int i = 0; i < MAX_SNOW; i++) {
    if (snow[i].active) {
      eraseSnowflake(snow[i].x, snow[i].y);
      snow[i].active = false;
    }
  }
  activeSnowCount = 0;
}

// Пересоздаём снежинки
void resetSnowflakes() {
  int potValue = analogRead(POT_PIN);
  int targetCount = map(potValue, 0, 4095, 0, MAX_SNOW);
  
  for (int i = 0; i < MAX_SNOW && activeSnowCount < targetCount; i++) {
    if (!snow[i].active) {
      createSnowflake(i);
      activeSnowCount++;
    }
  }
}

void drawCurrentImage() {
  const uint16_t* imageArray = showClock ? spasskaya_clock : spasskaya_tower;
  
  tft.startWrite();
  tft.setAddrWindow(0, 0, IMG_WIDTH, IMG_HEIGHT);
  
  for (uint32_t i = 0; i < IMG_PIXELS; i++) {
    tft.writeColor(pgm_read_word(&imageArray[i]), 1);
  }
  
  tft.endWrite();
}

// === 2. СИСТЕМА СНЕЖИНОК ===
void createSnowflake(int index) {
  snow[index].x = random(0, 238);
  snow[index].y = random(-30, -2);
  snow[index].speed = 0.4 + random(0, 16) / 10.0;
  snow[index].active = true;
}

// Стирание снежинки (восстановление фона из ТЕКУЩЕГО массива)
void eraseSnowflake(float x, float y) {
  int ix = (int)x;
  int iy = (int)y;
  
  // Берём ТЕКУЩИЙ активный массив
  const uint16_t* imageArray = showClock ? spasskaya_clock : spasskaya_tower;
  
  tft.startWrite();
  
  // Восстанавливаем 4 пикселя
  for (int dy = 0; dy < 2; dy++) {
    for (int dx = 0; dx < 2; dx++) {
      int px = ix + dx;
      int py = iy + dy;
      
      if (px >= 0 && px < IMG_WIDTH && py >= 0 && py < IMG_HEIGHT) {
        int imgIndex = py * IMG_WIDTH + px;
        uint16_t bgColor = pgm_read_word(&imageArray[imgIndex]);
        tft.writePixel(px, py, bgColor);
      }
    }
  }
  
  tft.endWrite();
}

// Рисование снежинки
void drawSnowflake(float x, float y) {
  int ix = (int)x;
  int iy = (int)y;
  
  tft.startWrite();
  
  for (int dy = 0; dy < 2; dy++) {
    for (int dx = 0; dx < 2; dx++) {
      int px = ix + dx;
      int py = iy + dy;
      
      if (px >= 0 && px < IMG_WIDTH && py >= 0 && py < IMG_HEIGHT) {
        tft.writePixel(px, py, SNOW_COLOR);
      }
    }
  }
  
  tft.endWrite();
}

void updateSnow() {
  // 1. Управление количеством через потенциометр
  int potValue = analogRead(POT_PIN);
  int targetCount = map(potValue, 0, 4095, 0, MAX_SNOW);
  
  // Добавляем снежинки, если нужно
  if (activeSnowCount < targetCount) {
    for (int i = 0; i < MAX_SNOW && activeSnowCount < targetCount; i++) {
      if (!snow[i].active) {
        createSnowflake(i);
        activeSnowCount++;
      }
    }
  }
  
  // Убираем снежинки, если нужно
  if (activeSnowCount > targetCount) {
    for (int i = 0; i < MAX_SNOW && activeSnowCount > targetCount; i++) {
      if (snow[i].active) {
        eraseSnowflake(snow[i].x, snow[i].y);
        snow[i].active = false;
        activeSnowCount--;
      }
    }
  }
  
  // 2. Двигаем и рисуем снежинки
  for (int i = 0; i < MAX_SNOW; i++) {
    if (snow[i].active) {
      // Запоминаем старую позицию
      float oldX = snow[i].x;
      float oldY = snow[i].y;
      
      // Двигаем
      snow[i].y += snow[i].speed;
      snow[i].x += sin(millis() * 0.0008 + i) * 0.4;
      
      // Проверка границ по X
      if (snow[i].x < -2) snow[i].x = 240;
      if (snow[i].x > 240) snow[i].x = -2;
      
      // Если упала ниже экрана
      if (snow[i].y > 322) {
        // Стираем на старой позиции
        eraseSnowflake(oldX, oldY);
        
        // Пересоздаём сверху
        snow[i].active = false;
        activeSnowCount--;
        
        if (activeSnowCount < targetCount) {
          createSnowflake(i);
          activeSnowCount++;
        }
      } else {
        // Стираем на старой позиции И рисуем на новой
        eraseSnowflake(oldX, oldY);
        drawSnowflake(snow[i].x, snow[i].y);
      }
    }
  }
  
  // 3. Информация в Serial
  static unsigned long lastReport = 0;
  if (millis() - lastReport > 3000) {
    Serial.print("Фон: ");
    Serial.print(showClock ? "Часы" : "Башня");
    Serial.print(" | Снежинок: ");
    Serial.print(activeSnowCount);
    Serial.print(" из ");
    Serial.println(targetCount);
    lastReport = millis();
  }
}